# 機能設計書 40-Grant URL検証

## 概要

本ドキュメントは、eShopアプリケーションにおけるGrant URL検証機能の設計仕様を定義するものである。本機能は、Webhooks.APIサービスが提供するWebhook管理機能の一部であり、Webhook登録時にGrant URLの有効性と所有権を検証するセキュリティ機能を提供する。

### 本機能の処理概要

本機能は、Webhookサブスクリプション登録時に、登録者が指定したWebhookエンドポイント（DestUrl）の所有権を検証する機能を提供する。HTTP OPTIONSリクエストとトークン照合により、悪意のある第三者が他人のサーバーにWebhook通知を送信させることを防止する。

**業務上の目的・背景**：Webhookは外部システムへの通知を行うため、悪用されると意図しない第三者のサーバーへ大量のリクエストを送信する攻撃（DDoS攻撃の踏み台）に悪用される可能性がある。本機能は、Webhookエンドポイントの所有者のみがサブスクリプションを登録できることを保証し、このようなセキュリティリスクを軽減する。Grant URLとDestUrlが同一オリジンであることを検証し、さらにトークンのエコーバックにより所有権を確認する。

**機能の利用シーン**：Webhook登録API（POST /api/webhooks）から呼び出され、登録リクエストの前処理として所有権検証を行う。検証に失敗した場合、サブスクリプションは登録されない。

**主要な処理内容**：
1. DestUrlとGrantUrlの同一オリジンチェック
2. Grant URLへのHTTP OPTIONSリクエスト送信
3. X-eshop-whtokenヘッダーでのトークン送信
4. レスポンスヘッダーからのトークン照合
5. 検証結果（true/false）の返却

**関連システム・外部連携**：Webhook登録機能（No.37）から呼び出される。外部のGrant URLエンドポイントへHTTP OPTIONSリクエストを送信する。

**権限による制御**：本機能は内部サービスとして動作し、直接的なAPIエンドポイントは公開されていない。Webhook登録APIの内部処理として使用される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 22 | Webhook追加画面 | 参照画面 | Webhook登録時のGrant URL検証 |

## 機能種別

セキュリティ機能 / 所有権検証

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| urlHook | string | Yes | Webhook送信先URL（DestUrl） | 有効なURL形式 |
| url | string | Yes | Grant検証用URL | 有効なURL形式、urlHookと同一オリジン |
| token | string | No | 認証用トークン | - |

### 入力データソース

- Webhook登録リクエスト: WebhookSubscriptionRequestから取得
- urlHook: request.Url
- url: request.GrantUrl
- token: request.Token

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 検証結果 | bool | 検証成功時true、失敗時false |

### 出力先

- Webhook登録API: 検証結果に基づき登録可否を判定

## 処理フロー

### 処理シーケンス

```
1. 同一オリジンチェック
   └─ CheckSameOrigin(urlHook, url)呼び出し
   └─ Scheme、Host、Portの一致を確認
   └─ 不一致の場合: false返却（Warning ログ）

2. HTTP OPTIONSリクエスト構築
   └─ new HttpRequestMessage(HttpMethod.Options, url)
   └─ X-eshop-whtokenヘッダーにトークン設定

3. リクエスト送信
   └─ HttpClient.SendAsync(msg)実行
   └─ 例外発生時: false返却（Warning ログ）

4. レスポンス検証
   └─ IsSuccessStatusCodeチェック
   └─ X-eshop-whtokenヘッダーの取得
   └─ 期待トークンとの照合

5. 検証結果返却
   └─ ステータス成功かつトークン一致: true
   └─ それ以外: false
```

### フローチャート

```mermaid
flowchart TD
    A[TestGrantUrl呼び出し] --> B[同一オリジンチェック]
    B --> C{同一オリジン?}
    C -->|No| D[Warningログ出力]
    D --> E[false返却]
    C -->|Yes| F[OPTIONSリクエスト構築]
    F --> G[X-eshop-whtokenヘッダー設定]
    G --> H[HttpClient.SendAsync]
    H --> I{例外発生?}
    I -->|Yes| J[Warningログ出力]
    J --> E
    I -->|No| K{ステータス成功?}
    K -->|No| L[false返却]
    K -->|Yes| M[レスポンスからトークン取得]
    M --> N{トークン一致?}
    N -->|Yes| O[true返却]
    N -->|No| L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-40-01 | 同一オリジン必須 | DestUrlとGrantUrlは同一オリジン（Scheme、Host、Port一致）である必要 | 全検証リクエスト |
| BR-40-02 | OPTIONSメソッド使用 | 検証にはHTTP OPTIONSメソッドを使用 | 全検証リクエスト |
| BR-40-03 | トークンエコーバック | 送信したトークンがレスポンスヘッダーで返却されることを検証 | トークン指定時 |
| BR-40-04 | 空トークンの扱い | トークンが空または未設定の場合、期待値はnull | トークン未指定時 |
| BR-40-05 | ステータスコード検証 | レスポンスの成功ステータス（2xx）を確認 | 全検証リクエスト |

### 計算ロジック

**同一オリジンチェック**:
```
同一オリジン = (urlHook.Scheme == url.Scheme) AND
               (urlHook.Host == url.Host) AND
               (urlHook.Port == url.Port)
```

**トークン照合ロジック**:
```
期待トークン = string.IsNullOrWhiteSpace(token) ? null : token
受信トークン = response.Headers["X-eshop-whtoken"].FirstOrDefault() ?? null
結果 = (期待トークン == 受信トークン)
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本機能ではデータベースアクセスなし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 同一オリジン違反 | urlHookとurlのオリジンが異なる | Warningログ、false返却 |
| - | HTTP通信エラー | ネットワーク障害、タイムアウト | Warningログ、false返却 |
| - | ステータスエラー | Grant URLが4xx/5xxを返却 | false返却 |
| - | トークン不一致 | エコーバックされたトークンが異なる | false返却 |

### リトライ仕様

- 本機能自体にはリトライ機能なし
- 呼び出し元のWebhook登録APIがユーザーに再試行を促す

## トランザクション仕様

- HTTP通信のみのため、トランザクション管理は不要

## パフォーマンス要件

- 検証処理: 5秒以内（外部通信を含む）
- タイムアウト設定はHttpClientのデフォルト

## セキュリティ考慮事項

- 同一オリジンポリシーによるなりすまし防止
- トークンエコーバックによる所有権確認
- HTTP OPTIONSメソッドによる安全なプローブ（データ変更なし）
- HTTPS通信推奨（Grant URLがHTTPSの場合）

## 備考

- HTTP OPTIONSはCORSプリフライトリクエストと同様の仕組みを利用
- Grant URLエンドポイントはX-eshop-whtokenヘッダーをエコーバックする必要がある
- IHttpClientFactoryを使用したHttpClient管理

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: インターフェースを理解する

サービスインターフェースを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | IGrantUrlTesterService.cs | `src/Webhooks.API/Services/IGrantUrlTesterService.cs` | 検証サービスインターフェース |

**読解のコツ**: TestGrantUrlメソッドの引数（urlHook, url, token）の役割を理解することが重要。

#### Step 2: 実装を理解する

GrantUrlTesterServiceの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | GrantUrlTesterService.cs | `src/Webhooks.API/Services/GrantUrlTesterService.cs` | 検証ロジックの実装 |

**主要処理フロー**:
- **3行目**: IHttpClientFactory、ILoggerのインジェクション
- **5-35行目**: TestGrantUrl - メイン検証処理
- **7-11行目**: CheckSameOrigin - 同一オリジンチェック
- **13-14行目**: HttpClientとOPTIONSリクエスト生成
- **15行目**: X-eshop-whtokenヘッダー設定
- **21行目**: HttpClient.SendAsync実行
- **22行目**: レスポンスからトークン取得
- **23行目**: 期待トークンの算出
- **27行目**: 成功ステータスとトークン一致を検証
- **37-45行目**: CheckSameOrigin - オリジン比較ロジック

#### Step 3: 呼び出し元を理解する

Webhook登録APIでの使用パターンを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | WebHooksApi.cs | `src/Webhooks.API/Apis/WebHooksApi.cs` | POST /での呼び出し |

**主要処理フロー**:
- **41行目**: `grantUrlTester.TestGrantUrl(request.Url, request.GrantUrl, request.Token ?? string.Empty)` - 検証呼び出し
- **43行目**: `if (grantOk)` - 検証結果による分岐

### プログラム呼び出し階層図

```
POST /api/webhooks
    │
    └─ IGrantUrlTesterService.TestGrantUrl(urlHook, url, token)
           │
           ├─ CheckSameOrigin(urlHook, url)
           │      │
           │      ├─ new Uri(urlHook).Scheme/Host/Port
           │      └─ new Uri(url).Scheme/Host/Port
           │
           ├─ new HttpRequestMessage(OPTIONS, url)
           │
           ├─ msg.Headers.Add("X-eshop-whtoken", token)
           │
           └─ HttpClient.SendAsync(msg)
                  │
                  ├─ response.IsSuccessStatusCode
                  │
                  └─ response.Headers["X-eshop-whtoken"]
                         │
                         └─ トークン照合 → true/false
```

### データフロー図

```
[入力]                        [処理]                         [出力]

urlHook (DestUrl)      ───▶ CheckSameOrigin()         ───▶ bool (true/false)
url (GrantUrl)               │
token                        ▼
                        同一オリジン検証
                             │
                             ▼
                        OPTIONSリクエスト構築
                        + X-eshop-whtoken設定
                             │
                             ▼
                        HttpClient.SendAsync()
                        (外部Grant URLへ)
                             │
                             ▼
                        レスポンス検証
                        ├─ ステータスコード
                        └─ X-eshop-whtokenエコーバック
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| GrantUrlTesterService.cs | `src/Webhooks.API/Services/GrantUrlTesterService.cs` | ソース | Grant URL検証サービス実装 |
| IGrantUrlTesterService.cs | `src/Webhooks.API/Services/IGrantUrlTesterService.cs` | ソース | 検証サービスインターフェース |
| WebHooksApi.cs | `src/Webhooks.API/Apis/WebHooksApi.cs` | ソース | 呼び出し元API |
| WebhookSubscriptionRequest.cs | `src/Webhooks.API/Model/WebhookSubscriptionRequest.cs` | ソース | リクエストモデル |
